<?php
/**
 * Database Configuration
 * Water Sort Game Admin Panel
 */

// Prevent direct access
if (!defined('ADMIN_ACCESS')) {
    die('Direct access denied');
}

// Include constants
require_once __DIR__ . '/constants.php';

/**
 * Database Connection Class
 * Uses PDO with secure configuration
 */
class Database {
    private static $instance = null;
    private $pdo;
    private $statement;
    
    /**
     * Get database instance (Singleton pattern)
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Private constructor to prevent direct instantiation
     */
    private function __construct() {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME;
            $this->pdo = new PDO($dsn, DB_USER, DB_PASS);
            $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $this->pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
            $this->pdo->setAttribute(PDO::ATTR_EMULATE_PREPARES, false);
            $this->pdo->setAttribute(PDO::MYSQL_ATTR_USE_BUFFERED_QUERY, true);
            $this->pdo->setAttribute(PDO::ATTR_PERSISTENT, true);
            $this->pdo->setAttribute(PDO::MYSQL_ATTR_INIT_COMMAND, "SET NAMES " . DB_CHARSET);
            $this->pdo->setAttribute(PDO::ATTR_TIMEOUT, 30);
            
        } catch (PDOException $e) {
            if (DEBUG_MODE) {
                die("Database connection failed: " . $e->getMessage());
            } else {
                die("Database connection failed. Please try again later.");
            }
        }
    }
    
    /**
     * Prepare SQL statement
     */
    public function prepare($sql) {
        $this->statement = $this->pdo->prepare($sql);
        return $this;
    }
    
    /**
     * Bind parameters to statement
     */
    public function bind($param, $value, $type = null) {
        if (is_null($type)) {
            switch (true) {
                case is_int($value):
                    $type = PDO::PARAM_INT;
                    break;
                case is_bool($value):
                    $type = PDO::PARAM_BOOL;
                    break;
                case is_null($value):
                    $type = PDO::PARAM_NULL;
                    break;
                default:
                    $type = PDO::PARAM_STR;
            }
        }
        $this->statement->bindValue($param, $value, $type);
        return $this;
    }
    
    /**
     * Execute prepared statement
     */
    public function execute() {
        try {
            return $this->statement->execute();
        } catch (PDOException $e) {
            if (DEBUG_MODE) {
                die("Query execution failed: " . $e->getMessage());
            } else {
                error_log("Database error: " . $e->getMessage());
                return false;
            }
        }
    }
    
    /**
     * Get multiple records
     */
    public function getAll() {
        $this->execute();
        return $this->statement->fetchAll();
    }
    
    /**
     * Get single record
     */
    public function get() {
        $this->execute();
        return $this->statement->fetch();
    }
    
    /**
     * Get single column value
     */
    public function fetchColumn($column = 0) {
        $this->execute();
        return $this->statement->fetchColumn($column);
    }
    
    /**
     * Get row count
     */
    public function rowCount() {
        return $this->statement->rowCount();
    }
    
    /**
     * Get last inserted ID
     */
    public function lastInsertId() {
        return $this->pdo->lastInsertId();
    }
    
    /**
     * Begin transaction
     */
    public function beginTransaction() {
        return $this->pdo->beginTransaction();
    }
    
    /**
     * Commit transaction
     */
    public function commit() {
        return $this->pdo->commit();
    }
    
    /**
     * Rollback transaction
     */
    public function rollBack() {
        return $this->pdo->rollBack();
    }
    
    /**
     * Get PDO instance for advanced operations
     */
    public function getPdo() {
        return $this->pdo;
    }
}

/**
 * Quick query function for simple operations
 */
function query($sql, $params = []) {
    $db = Database::getInstance();
    $db->prepare($sql);
    
    foreach ($params as $key => $value) {
        if (is_int($key)) {
            $db->bind($key + 1, $value);
        } else {
            $db->bind($key, $value);
        }
    }
    
    return $db;
}

/**
 * Quick fetch function
 */
function fetch($sql, $params = []) {
    return query($sql, $params)->get();
}

/**
 * Quick fetch all function
 */
function fetchAll($sql, $params = []) {
    return query($sql, $params)->getAll();
}

/**
 * Quick insert function
 */
function insert($table, $data) {
    $columns = array_keys($data);
    $placeholders = array_fill(0, count($columns), '?');
    
    $sql = "INSERT INTO {$table} (" . implode(', ', $columns) . ") 
            VALUES (" . implode(', ', $placeholders) . ")";
    
    $db = query($sql, array_values($data));
    return $db->lastInsertId();
}

/**
 * Quick update function
 */
function update($table, $data, $where, $whereParams = []) {
    $setParts = [];
    $params = [];
    
    foreach ($data as $column => $value) {
        $setParts[] = "{$column} = ?";
        $params[] = $value;
    }
    
    $sql = "UPDATE {$table} SET " . implode(', ', $setParts) . " WHERE {$where}";
    $params = array_merge($params, $whereParams);
    
    $db = query($sql, $params);
    return $db->rowCount();
}

/**
 * Quick delete function
 */
function delete($table, $where, $params = []) {
    $sql = "DELETE FROM {$table} WHERE {$where}";
    $db = query($sql, $params);
    return $db->rowCount();
}
?>
