<?php
/**
 * Security Functions
 * Water Sort Game Admin Panel
 */

// Include required files
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/functions.php';

/**
 * Session Management Class
 */
class SessionManager {
    
    /**
     * Initialize secure session
     */
    public static function init() {
        // Set session parameters
        ini_set('session.cookie_httponly', COOKIE_HTTPONLY);
        ini_set('session.use_only_cookies', 1);
        ini_set('session.cookie_secure', COOKIE_SECURE);
        ini_set('session.gc_maxlifetime', SESSION_LIFETIME);
        ini_set('session.cookie_samesite', 'Strict');
        
        // Set session name
        session_name(SESSION_NAME);
        
        // Start session
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        
        // Regenerate session ID periodically
        if (!isset($_SESSION['last_regeneration']) || 
            time() - $_SESSION['last_regeneration'] > 300) { // 5 minutes
            self::regenerate();
        }
        
        // Check session timeout
        self::checkTimeout();
    }
    
    /**
     * Regenerate session ID
     */
    public static function regenerate() {
        session_regenerate_id(true);
        $_SESSION['last_regeneration'] = time();
    }
    
    /**
     * Check session timeout
     */
    public static function checkTimeout() {
        if (isset($_SESSION['last_activity']) && 
            time() - $_SESSION['last_activity'] > SESSION_LIFETIME) {
            self::destroy();
            redirect('auth/login.php?timeout=1');
        }
        $_SESSION['last_activity'] = time();
    }
    
    /**
     * Destroy session
     */
    public static function destroy() {
        $_SESSION = array();
        
        if (ini_get("session.use_cookies")) {
            $params = session_get_cookie_params();
            setcookie(
                session_name(), 
                '', 
                time() - 42000,
                $params["path"], 
                $params["domain"],
                $params["secure"], 
                $params["httponly"]
            );
        }
        
        session_destroy();
    }
    
    /**
     * Check if user is logged in
     */
    public static function isLoggedIn() {
        return isset($_SESSION['admin_id']) && 
               !empty($_SESSION['admin_id']) && 
               isset($_SESSION['admin_username']);
    }
    
    /**
     * Require login to access page
     */
    public static function requireLogin() {
        if (!self::isLoggedIn()) {
            $_SESSION['redirect_url'] = getCurrentURL();
            redirect('auth/login.php');
        }
        self::checkTimeout();
    }
    
    /**
     * Get current admin user
     */
    public static function getCurrentAdmin() {
        if (!self::isLoggedIn()) {
            return null;
        }
        
        return [
            'id' => $_SESSION['admin_id'],
            'username' => $_SESSION['admin_username'],
            'email' => $_SESSION['admin_email'] ?? '',
            'role' => $_SESSION['admin_role'] ?? 'admin',
            'login_time' => $_SESSION['login_time'] ?? time()
        ];
    }
    
    /**
     * Set admin session data
     */
    public static function setAdmin($admin) {
        $_SESSION['admin_id'] = $admin['id'];
        $_SESSION['admin_username'] = $admin['username'];
        $_SESSION['admin_email'] = $admin['email'];
        $_SESSION['admin_role'] = $admin['role'];
        $_SESSION['login_time'] = time();
        $_SESSION['last_activity'] = time();
        $_SESSION['last_regeneration'] = time();
    }
}

/**
 * Authentication Class
 */
class Auth {
    
    /**
     * Check if user is logged in
     */
    public static function isLoggedIn() {
        return SessionManager::isLoggedIn();
    }
    
    /**
     * Login user
     */
    public static function login($username, $password, $remember = false) {
        $result = self::attempt($username, $password);
        
        if ($result['success'] && $remember) {
            // Set remember me cookie (30 days)
            $token = generateRandomToken(32);
            $expiry = time() + (30 * 24 * 60 * 60);
            
            setcookie('admin_remember', $token, $expiry, '/', '', false, true);
            
            // Store token in database (you'd need to implement this)
            // update('admins', ['remember_token' => $token], 'username = ?', [$username]);
        }
        
        return $result;
    }
    
    /**
     * Attempt login
     */
    public static function attempt($username, $password) {
        $username = sanitize($username);
        
        // Check if account is locked
        if (self::isAccountLocked($username)) {
            return ['success' => false, 'message' => 'Account is temporarily locked. Please try again later.'];
        }
        
        // Get admin user
        $admin = fetch("SELECT * FROM admins WHERE username = ?", [$username]);
        
        if (!$admin) {
            self::recordFailedAttempt($username);
            return ['success' => false, 'message' => 'Invalid username or password'];
        }
        
        // Check password
        if (!verifyPassword($password, $admin['password'])) {
            self::recordFailedAttempt($username);
            return ['success' => false, 'message' => 'Invalid username or password'];
        }
        
        // Check if account is locked
        if ($admin['locked_until'] && strtotime($admin['locked_until']) > time()) {
            return ['success' => false, 'message' => 'Account is temporarily locked. Please try again later.'];
        }
        
        // Successful login
        self::clearFailedAttempts($username);
        
        // Update last login
        update('admins', [
            'last_login' => date('Y-m-d H:i:s'),
            'login_attempts' => 0,
            'locked_until' => null
        ], 'id = ?', [$admin['id']]);
        
        // Log activity
        logActivity('login', 'admins', $admin['id'], null, [
            'ip_address' => getClientIP(),
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? ''
        ]);
        
        // Set session
        SessionManager::setAdmin($admin);
        
        return ['success' => true, 'message' => 'Login successful'];
    }
    
    /**
     * Logout
     */
    public static function logout() {
        $admin = SessionManager::getCurrentAdmin();
        
        if ($admin) {
            // Log activity
            logActivity('logout', 'admins', $admin['id'], null, [
                'ip_address' => getClientIP()
            ]);
        }
        
        SessionManager::destroy();
    }
    
    /**
     * Check if account is locked
     */
    private static function isAccountLocked($username) {
        $admin = fetch("SELECT locked_until FROM admins WHERE username = ?", [$username]);
        
        if (!$admin || !$admin['locked_until']) {
            return false;
        }
        
        return strtotime($admin['locked_until']) > time();
    }
    
    /**
     * Record failed login attempt
     */
    private static function recordFailedAttempt($username) {
        $admin = fetch("SELECT id, login_attempts FROM admins WHERE username = ?", [$username]);
        
        if ($admin) {
            $attempts = $admin['login_attempts'] + 1;
            $lockedUntil = null;
            
            if ($attempts >= MAX_LOGIN_ATTEMPTS) {
                $lockedUntil = date('Y-m-d H:i:s', time() + LOCKOUT_DURATION);
            }
            
            update('admins', [
                'login_attempts' => $attempts,
                'locked_until' => $lockedUntil
            ], 'id = ?', [$admin['id']]);
        }
        
        // Log failed attempt
        logActivity('login_failed', 'admins', $admin['id'] ?? null, null, [
            'username' => $username,
            'ip_address' => getClientIP()
        ]);
    }
    
    /**
     * Clear failed login attempts
     */
    private static function clearFailedAttempts($username) {
        update('admins', [
            'login_attempts' => 0,
            'locked_until' => null
        ], 'username = ?', [$username]);
    }
}

/**
 * Activity Logging
 */
function logActivity($action, $table = null, $recordId = null, $oldValues = null, $newValues = null) {
    $admin = SessionManager::getCurrentAdmin();
    
    $data = [
        'admin_id' => $admin['id'] ?? null,
        'action' => $action,
        'table_name' => $table,
        'record_id' => $recordId,
        'old_values' => $oldValues ? safeJSONEncode($oldValues) : null,
        'new_values' => $newValues ? safeJSONEncode($newValues) : null,
        'ip_address' => $newValues['ip_address'] ?? getClientIP(),
        'user_agent' => $newValues['user_agent'] ?? ($_SERVER['HTTP_USER_AGENT'] ?? '')
    ];
    
    insert('activity_logs', $data);
}

/**
 * Input Validation
 */
class Validator {
    
    /**
     * Validate login form
     */
    public static function validateLogin($data) {
        $errors = [];
        
        if (empty($data['username'])) {
            $errors['username'] = 'Username is required';
        } elseif (!validateLength($data['username'], 3, 50)) {
            $errors['username'] = 'Username must be between 3 and 50 characters';
        }
        
        if (empty($data['password'])) {
            $errors['password'] = 'Password is required';
        } elseif (!validateLength($data['password'], PASSWORD_MIN_LENGTH, 100)) {
            $errors['password'] = 'Password must be at least ' . PASSWORD_MIN_LENGTH . ' characters';
        }
        
        // CSRF token validation
        if (!isset($data['csrf_token']) || !verifyCSRFToken($data['csrf_token'])) {
            $errors['csrf'] = 'Invalid request token';
        }
        
        return $errors;
    }
    
    /**
     * Validate user data
     */
    public static function validateUser($data, $isEdit = false) {
        $errors = [];
        
        if (empty($data['username'])) {
            $errors['username'] = 'Username is required';
        } elseif (!validateLength($data['username'], 3, 50)) {
            $errors['username'] = 'Username must be between 3 and 50 characters';
        } elseif (!preg_match('/^[a-zA-Z0-9_]+$/', $data['username'])) {
            $errors['username'] = 'Username can only contain letters, numbers, and underscores';
        }
        
        if (empty($data['email'])) {
            $errors['email'] = 'Email is required';
        } elseif (!validateEmail($data['email'])) {
            $errors['email'] = 'Invalid email format';
        }
        
        if (!$isEdit && empty($data['password'])) {
            $errors['password'] = 'Password is required';
        } elseif (isset($data['password']) && !empty($data['password']) && 
                !validateLength($data['password'], PASSWORD_MIN_LENGTH, 100)) {
            $errors['password'] = 'Password must be at least ' . PASSWORD_MIN_LENGTH . ' characters';
        }
        
        if (isset($data['coins']) && !validateNumeric($data['coins'], 0, 999999)) {
            $errors['coins'] = 'Coins must be a number between 0 and 999,999';
        }
        
        if (isset($data['hints']) && !validateNumeric($data['hints'], 0, 999)) {
            $errors['hints'] = 'Hints must be a number between 0 and 999';
        }
        
        if (isset($data['undos']) && !validateNumeric($data['undos'], 0, 999)) {
            $errors['undos'] = 'Undos must be a number between 0 and 999';
        }
        
        return $errors;
    }
    
    /**
     * Validate level data
     */
    public static function validateLevel($data, $isEdit = false) {
        $errors = [];
        
        if (empty($data['name'])) {
            $errors['name'] = 'Level name is required';
        } elseif (!validateLength($data['name'], 1, 100)) {
            $errors['name'] = 'Level name must be between 1 and 100 characters';
        }
        
        if (isset($data['level_number']) && !validateNumeric($data['level_number'], 1, 9999)) {
            $errors['level_number'] = 'Level number must be between 1 and 9999';
        }
        
        if (isset($data['difficulty']) && !in_array($data['difficulty'], ['Easy', 'Medium', 'Hard', 'Expert'])) {
            $errors['difficulty'] = 'Invalid difficulty level';
        }
        
        if (isset($data['bottles_count']) && !validateNumeric($data['bottles_count'], 3, 20)) {
            $errors['bottles_count'] = 'Bottles count must be between 3 and 20';
        }
        
        if (isset($data['colors_count']) && !validateNumeric($data['colors_count'], 2, 10)) {
            $errors['colors_count'] = 'Colors count must be between 2 and 10';
        }
        
        if (isset($data['bottle_capacity']) && !validateNumeric($data['bottle_capacity'], 3, 10)) {
            $errors['bottle_capacity'] = 'Bottle capacity must be between 3 and 10';
        }
        
        if (empty($data['initial_state'])) {
            $errors['initial_state'] = 'Initial state is required';
        } else {
            $json = safeJSONDecode($data['initial_state']);
            if ($json === false) {
                $errors['initial_state'] = 'Initial state must be valid JSON';
            } elseif (!is_array($json) || empty($json)) {
                $errors['initial_state'] = 'Initial state must be a non-empty array';
            }
        }
        
        return $errors;
    }
}

// Initialize session (only if not already started)
if (session_status() === PHP_SESSION_NONE) {
    SessionManager::init();
}
?>
