<?php
// Define admin access for security
define('ADMIN_ACCESS', true);

// Set page title
$page_title = 'Activity Logs';

// Set current page for navigation
$current_page = 'activity-logs.php';

// Include required files
require_once __DIR__ . '/../config/constants.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/functions.php';
require_once __DIR__ . '/../includes/security.php';

// Include header
include_once ADMIN_PATH . '/includes/header.php';

// Handle filters
$actionFilter = sanitizeInput($_GET['action_filter'] ?? '');
$adminFilter = sanitizeInput($_GET['admin_filter'] ?? '');
$dateFilter = sanitizeInput($_GET['date_filter'] ?? '7'); // 7, 30, 90 days

// Build WHERE clause
$where = [];
$params = [];

// Action filter
if (!empty($actionFilter)) {
    $where[] = "al.action LIKE ?";
    $params[] = "%$actionFilter%";
}

// Admin filter
if (!empty($adminFilter)) {
    $where[] = "a.username LIKE ?";
    $params[] = "%$adminFilter%";
}

// Date filter
if ($dateFilter === '7') {
    $where[] = "DATE(al.created_at) >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)";
} elseif ($dateFilter === '30') {
    $where[] = "DATE(al.created_at) >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)";
} elseif ($dateFilter === '90') {
    $where[] = "DATE(al.created_at) >= DATE_SUB(CURDATE(), INTERVAL 90 DAY)";
}

$whereClause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';

// Pagination
$page = max(1, intval($_GET['page'] ?? 1));
$limit = ITEMS_PER_PAGE;
$offset = ($page - 1) * $limit;

// Get total records
$countQuery = "SELECT COUNT(*) as total 
               FROM activity_logs al 
               LEFT JOIN admins a ON al.admin_id = a.id 
               $whereClause";
$totalRecords = fetch($countQuery, $params)['total'];
$totalPages = ceil($totalRecords / $limit);

// Get activity logs
$query = "SELECT al.*, 
                a.username as admin_username
         FROM activity_logs al
         LEFT JOIN admins a ON al.admin_id = a.id
         $whereClause
         ORDER BY al.created_at DESC
         LIMIT ? OFFSET ?";

$logsParams = array_merge($params, [$limit, $offset]);
$activityLogs = fetchAll($query, $logsParams);

// Get action statistics
$actionStats = fetchAll("
    SELECT action, COUNT(*) as count
    FROM activity_logs
    GROUP BY action
    ORDER BY count DESC
    LIMIT 10
");

// Get recent activity summary
$summaryQuery = "SELECT 
                    COUNT(*) as total_actions,
                    COUNT(DISTINCT admin_id) as unique_admins,
                    COUNT(CASE WHEN action LIKE '%login%' THEN 1 END) as logins,
                    COUNT(CASE WHEN action LIKE '%delete%' THEN 1 END) as deletions,
                    COUNT(CASE WHEN action LIKE '%add%' OR action LIKE '%create%' THEN 1 END) as creations
                FROM activity_logs al
                LEFT JOIN admins a ON al.admin_id = a.id 
                $whereClause";

$summary = fetch($summaryQuery, $params);
?>

<!-- Activity Logs Content -->
<div class="activity-logs-content">
    <!-- Page Header -->
    <div class="page-header">
        <h1 class="page-title">Activity Logs</h1>
        <p class="page-description">Monitor all admin actions and system activities</p>
    </div>
    
    <!-- Summary Cards -->
    <div class="summary-cards">
        <div class="summary-card">
            <div class="summary-icon">
                <span class="icon">📊</span>
            </div>
            <div class="summary-content">
                <h3 class="summary-title">Total Actions</h3>
                <p class="summary-value"><?php echo number_format($summary['total_actions'] ?? 0); ?></p>
            </div>
        </div>
        
        <div class="summary-card">
            <div class="summary-icon">
                <span class="icon">👤</span>
            </div>
            <div class="summary-content">
                <h3 class="summary-title">Active Admins</h3>
                <p class="summary-value"><?php echo number_format($summary['unique_admins'] ?? 0); ?></p>
            </div>
        </div>
        
        <div class="summary-card">
            <div class="summary-icon">
                <span class="icon">👥</span>
            </div>
            <div class="summary-content">
                <h3 class="summary-title">Affected Users</h3>
                <p class="summary-value">N/A</p>
            </div>
        </div>
        
        <div class="summary-card">
            <div class="summary-icon">
                <span class="icon">🔐</span>
            </div>
            <div class="summary-content">
                <h3 class="summary-title">Logins</h3>
                <p class="summary-value"><?php echo number_format($summary['logins'] ?? 0); ?></p>
            </div>
        </div>
    </div>
    
    <!-- Filters -->
    <div class="filters-section">
        <form method="GET" class="filters-form">
            <div class="filter-row">
                <div class="filter-group">
                    <label for="action_filter" class="filter-label">Action</label>
                    <input type="text" 
                           id="action_filter" 
                           name="action_filter" 
                           class="filter-input" 
                           placeholder="Search actions..."
                           value="<?php echo htmlspecialchars($actionFilter); ?>">
                </div>
                
                <div class="filter-group">
                    <label for="admin_filter" class="filter-label">Admin</label>
                    <input type="text" 
                           id="admin_filter" 
                           name="admin_filter" 
                           class="filter-input" 
                           placeholder="Admin username..."
                           value="<?php echo htmlspecialchars($adminFilter); ?>">
                </div>
                
                <div class="filter-group">
                    <label for="date_filter" class="filter-label">Date Range</label>
                    <select id="date_filter" name="date_filter" class="filter-select">
                        <option value="7" <?php echo $dateFilter === '7' ? 'selected' : ''; ?>>Last 7 Days</option>
                        <option value="30" <?php echo $dateFilter === '30' ? 'selected' : ''; ?>>Last 30 Days</option>
                        <option value="90" <?php echo $dateFilter === '90' ? 'selected' : ''; ?>>Last 90 Days</option>
                    </select>
                </div>
                
                <div class="filter-actions">
                    <button type="submit" class="btn btn-primary">
                        <span class="btn-icon">🔍</span>
                        Apply Filters
                    </button>
                    <a href="<?php echo ADMIN_URL; ?>/modules/activity-logs.php" class="btn btn-secondary">
                        <span class="btn-icon">✖</span>
                        Clear
                    </a>
                    <button type="button" class="btn btn-secondary" onclick="exportLogs()">
                        <span class="btn-icon">📥</span>
                        Export
                    </button>
                    <button type="button" class="btn btn-danger" onclick="clearLogs()">
                        <span class="btn-icon">🗑️</span>
                        Clear Old Logs
                    </button>
                </div>
            </div>
        </form>
    </div>
    
    <!-- Activity Logs Grid -->
    <div class="logs-grid">
        <!-- Main Logs Table -->
        <div class="logs-main">
            <div class="table-container">
                <div class="table-header">
                    <h2 class="table-title">Activity Logs (<?php echo number_format($totalRecords); ?>)</h2>
                    <div class="table-actions">
                        <button class="btn btn-sm btn-secondary" onclick="refreshLogs()">
                            <span class="btn-icon">🔄</span>
                            Refresh
                        </button>
                    </div>
                </div>
                
                <?php if (empty($activityLogs)): ?>
                <div class="empty-state">
                    <div class="empty-icon">📝</div>
                    <h3>No activity logs found</h3>
                    <p>No activity logs match your current filters.</p>
                </div>
                <?php else: ?>
                <div class="data-table">
                    <table class="logs-table">
                        <thead>
                            <tr>
                                <th>Date/Time</th>
                                <th>Action</th>
                                <th>Admin</th>
                                <th>Details</th>
                                <th>IP Address</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($activityLogs as $log): ?>
                            <tr>
                                <td>
                                    <div class="datetime-info">
                                        <div class="date"><?php echo formatDate($log['created_at'], 'M j, Y'); ?></div>
                                        <div class="time"><?php echo formatDate($log['created_at'], 'g:i A'); ?></div>
                                    </div>
                                </td>
                                <td>
                                    <span class="action-badge action-<?php echo getActionClass($log['action']); ?>">
                                        <?php echo formatActionName($log['action']); ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="admin-info">
                                        <div class="admin-name"><?php echo htmlspecialchars($log['admin_username'] ?? 'System'); ?></div>
                                        <?php if ($log['admin_id']): ?>
                                        <div class="admin-id">ID: #<?php echo $log['admin_id']; ?></div>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td>
                                    <div class="log-details">
                                        <?php if ($log['details']): ?>
                                        <div class="details-text"><?php echo htmlspecialchars($log['details']); ?></div>
                                        <?php endif; ?>
                                        
                                        <?php if ($log['table_name']): ?>
                                        <div class="table-info">
                                            <span class="table-name">Table: <?php echo htmlspecialchars($log['table_name']); ?></span>
                                            <?php if ($log['record_id']): ?>
                                            <span class="record-id">ID: #<?php echo $log['record_id']; ?></span>
                                            <?php endif; ?>
                                        </div>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td>
                                    <span class="ip-address"><?php echo htmlspecialchars($log['ip_address'] ?? 'Unknown'); ?></span>
                                </td>
                                <td>
                                    <div class="action-buttons">
                                        <button class="btn btn-sm btn-info" 
                                                onclick="viewLogDetails(<?php echo $log['id']; ?>)"
                                                title="View Details">
                                            <span class="btn-icon">👁️</span>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                
                <!-- Pagination -->
                <?php if ($totalPages > 1): ?>
                <div class="pagination">
                    <?php
                    $currentUrl = ADMIN_URL . '/modules/activity-logs.php';
                    $queryParams = [];
                    if (!empty($actionFilter)) $queryParams['action_filter'] = $actionFilter;
                    if (!empty($adminFilter)) $queryParams['admin_filter'] = $adminFilter;
                    if ($dateFilter !== '7') $queryParams['date_filter'] = $dateFilter;
                    
                    $urlString = $currentUrl . (!empty($queryParams) ? '?' . http_build_query($queryParams) : '');
                    ?>
                    
                    <?php if ($page > 1): ?>
                    <a href="<?php echo $urlString . (strpos($urlString, '?') !== false ? '&' : '?') . 'page=' . ($page - 1); ?>" 
                       class="btn btn-pagination">« Previous</a>
                    <?php endif; ?>
                    
                    <?php
                    $startPage = max(1, $page - 2);
                    $endPage = min($totalPages, $page + 2);
                    
                    for ($i = $startPage; $i <= $endPage; $i++):
                    ?>
                    <a href="<?php echo $urlString . (strpos($urlString, '?') !== false ? '&' : '?') . 'page=' . $i; ?>" 
                       class="btn btn-pagination <?php echo $i == $page ? 'active' : ''; ?>">
                        <?php echo $i; ?>
                    </a>
                    <?php endfor; ?>
                    
                    <?php if ($page < $totalPages): ?>
                    <a href="<?php echo $urlString . (strpos($urlString, '?') !== false ? '&' : '?') . 'page=' . ($page + 1); ?>" 
                       class="btn btn-pagination">Next »</a>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Sidebar -->
        <div class="logs-sidebar">
            <!-- Action Statistics -->
            <div class="sidebar-card">
                <h3 class="sidebar-title">📈 Action Statistics</h3>
                <div class="action-stats">
                    <?php foreach ($actionStats as $stat): ?>
                    <div class="stat-item">
                        <div class="stat-action"><?php echo formatActionName($stat['action']); ?></div>
                        <div class="stat-count"><?php echo number_format($stat['count']); ?></div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
            
            <!-- Recent Activity Types -->
            <div class="sidebar-card">
                <h3 class="sidebar-title">🏷️ Activity Types</h3>
                <div class="activity-types">
                    <div class="type-item">
                        <span class="type-badge action-login">🔐 Login</span>
                        <span class="type-desc">Admin authentication</span>
                    </div>
                    <div class="type-item">
                        <span class="type-badge action-create">➕ Create</span>
                        <span class="type-desc">Add new records</span>
                    </div>
                    <div class="type-item">
                        <span class="type-badge action-edit">✏️ Edit</span>
                        <span class="type-desc">Update records</span>
                    </div>
                    <div class="type-item">
                        <span class="type-badge action-delete">🗑️ Delete</span>
                        <span class="type-desc">Remove records</span>
                    </div>
                    <div class="type-item">
                        <span class="type-badge action-settings">⚙️ Settings</span>
                        <span class="type-desc">Configuration changes</span>
                    </div>
                </div>
            </div>
            
            <!-- Quick Filters -->
            <div class="sidebar-card">
                <h3 class="sidebar-title">⚡ Quick Filters</h3>
                <div class="quick-filters">
                    <a href="?action_filter=login" class="filter-link">
                        <span class="filter-icon">🔐</span>
                        Logins Only
                    </a>
                    <a href="?action_filter=delete" class="filter-link">
                        <span class="filter-icon">🗑️</span>
                        Deletions Only
                    </a>
                    <a href="?action_filter=create" class="filter-link">
                        <span class="filter-icon">➕</span>
                        Creations Only
                    </a>
                    <a href="?date_filter=1" class="filter-link">
                        <span class="filter-icon">📅</span>
                        Today Only
                    </a>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Log Details Modal -->
<div id="logDetailsModal" class="modal" style="display: none;">
    <div class="modal-content modal-large">
        <div class="modal-header">
            <h3>Activity Log Details</h3>
            <button class="modal-close" onclick="closeDetailsModal()">&times;</button>
        </div>
        <div class="modal-body">
            <div id="logDetailsContent">
                <!-- Details will be loaded here -->
            </div>
        </div>
        <div class="modal-footer">
            <button class="btn btn-secondary" onclick="closeDetailsModal()">Close</button>
        </div>
    </div>
</div>

<!-- Clear Logs Confirmation Modal -->
<div id="clearLogsModal" class="modal" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Clear Old Activity Logs</h3>
            <button class="modal-close" onclick="closeClearModal()">&times;</button>
        </div>
        <div class="modal-body">
            <p>Select how old logs to clear:</p>
            <div class="clear-options">
                <label class="radio-label">
                    <input type="radio" name="clear_period" value="30" checked>
                    <span class="radio-text">Older than 30 days</span>
                </label>
                <label class="radio-label">
                    <input type="radio" name="clear_period" value="90">
                    <span class="radio-text">Older than 90 days</span>
                </label>
                <label class="radio-label">
                    <input type="radio" name="clear_period" value="365">
                    <span class="radio-text">Older than 1 year</span>
                </label>
            </div>
            <p class="warning-text">⚠️ This action cannot be undone!</p>
        </div>
        <div class="modal-footer">
            <button class="btn btn-secondary" onclick="closeClearModal()">Cancel</button>
            <button class="btn btn-danger" onclick="confirmClearLogs()">Clear Logs</button>
        </div>
    </div>
</div>

<script>
function viewLogDetails(logId) {
    fetch('<?php echo ADMIN_URL; ?>/modules/log-details.php?id=' + logId)
        .then(response => response.text())
        .then(html => {
            document.getElementById('logDetailsContent').innerHTML = html;
            document.getElementById('logDetailsModal').style.display = 'block';
        })
        .catch(error => {
            console.error('Error loading log details:', error);
            showNotification('Error loading log details', 'error');
        });
}

function closeDetailsModal() {
    document.getElementById('logDetailsModal').style.display = 'none';
}

function refreshLogs() {
    location.reload();
}

function exportLogs() {
    const params = new URLSearchParams(window.location.search);
    params.set('export', '1');
    window.location.href = '<?php echo ADMIN_URL; ?>/modules/export-logs.php?' + params.toString();
}

function clearLogs() {
    document.getElementById('clearLogsModal').style.display = 'block';
}

function closeClearModal() {
    document.getElementById('clearLogsModal').style.display = 'none';
}

function confirmClearLogs() {
    const period = document.querySelector('input[name="clear_period"]:checked').value;
    
    if (!confirm(`Are you sure you want to delete all activity logs older than ${period} days?`)) return;
    
    const formData = new FormData();
    formData.append('period', period);
    formData.append('csrf_token', '<?php echo generateCSRFToken(); ?>');
    
    fetch('<?php echo ADMIN_URL; ?>/modules/clear-logs.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('Old logs cleared successfully', 'success');
            closeClearModal();
            location.reload();
        } else {
            showNotification(data.message || 'Error clearing logs', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('Error clearing logs', 'error');
    });
}

// Close modals when clicking outside
window.onclick = function(event) {
    const detailsModal = document.getElementById('logDetailsModal');
    const clearModal = document.getElementById('clearLogsModal');
    
    if (event.target == detailsModal) {
        closeDetailsModal();
    }
    if (event.target == clearModal) {
        closeClearModal();
    }
}

// Helper functions
function getActionClass(action) {
    if (action.includes('login')) return 'login';
    if (action.includes('create') || action.includes('add')) return 'create';
    if (action.includes('edit') || action.includes('update')) return 'edit';
    if (action.includes('delete')) return 'delete';
    if (action.includes('settings')) return 'settings';
    return 'other';
}

function formatActionName(action) {
    return action.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase());
}
</script>

<?php
// Helper functions
function getActionClass($action) {
    if (strpos($action, 'login') !== false) return 'login';
    if (strpos($action, 'create') !== false || strpos($action, 'add') !== false) return 'create';
    if (strpos($action, 'edit') !== false || strpos($action, 'update') !== false) return 'edit';
    if (strpos($action, 'delete') !== false) return 'delete';
    if (strpos($action, 'settings') !== false) return 'settings';
    return 'other';
}

function formatActionName($action) {
    return ucwords(str_replace('_', ' ', $action));
}

// Include footer
include_once ADMIN_PATH . '/includes/footer.php';
?>
