<?php
// Define admin access for security
define('ADMIN_ACCESS', true);

// Set page title
$page_title = 'Edit Level';

// Set current page for navigation
$current_page = 'edit-level.php';

// Include required files
require_once __DIR__ . '/../config/constants.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/functions.php';
require_once __DIR__ . '/../includes/security.php';

// Include header
include_once ADMIN_PATH . '/includes/header.php';

// Get level ID
$levelId = intval($_GET['id'] ?? 0);
if ($levelId <= 0) {
    showNotification('Invalid level ID', 'error');
    header('Location: ' . ADMIN_URL . '/modules/levels.php');
    exit;
}

// Get level data
$level = fetch("SELECT * FROM levels WHERE id = ?", [$levelId]);
if (!$level) {
    showNotification('Level not found', 'error');
    header('Location: ' . ADMIN_URL . '/modules/levels.php');
    exit;
}

// Initialize variables
$errors = [];
$success = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate CSRF token
    if (!isset($_POST['csrf_token']) || !verifyCSRFToken($_POST['csrf_token'])) {
        $errors[] = 'Invalid request token';
    } else {
        // Sanitize and validate inputs
        $name = sanitizeInput($_POST['name'] ?? '');
        $description = sanitizeInput($_POST['description'] ?? '');
        $difficulty = sanitizeInput($_POST['difficulty'] ?? 'Easy');
        $bottlesCount = intval($_POST['bottles_count'] ?? 4);
        $colorsCount = intval($_POST['colors_count'] ?? 2);
        $bottleCapacity = intval($_POST['bottle_capacity'] ?? 4);
        $solutionMoves = intval($_POST['solution_moves'] ?? 0);
        $timeLimit = intval($_POST['time_limit'] ?? 0);
        $rewardCoins = intval($_POST['reward_coins'] ?? 10);
        $isActive = isset($_POST['is_active']) ? 1 : 0;
        
        // Validate required fields
        if (empty($name)) {
            $errors['name'] = 'Level name is required';
        }
        
        if ($bottlesCount < 3 || $bottlesCount > 20) {
            $errors['bottles_count'] = 'Bottles count must be between 3 and 20';
        }
        
        if ($colorsCount < 2 || $colorsCount > 10) {
            $errors['colors_count'] = 'Colors count must be between 2 and 10';
        }
        
        if ($bottleCapacity < 3 || $bottleCapacity > 10) {
            $errors['bottle_capacity'] = 'Bottle capacity must be between 3 and 10';
        }
        
        if (!in_array($difficulty, ['Easy', 'Medium', 'Hard', 'Expert'])) {
            $errors['difficulty'] = 'Invalid difficulty level';
        }
        
        // Validate initial state JSON
        $initialState = $_POST['initial_state'] ?? '';
        if (empty($initialState)) {
            $errors['initial_state'] = 'Initial state is required';
        } else {
            $decoded = json_decode($initialState, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                $errors['initial_state'] = 'Invalid JSON format: ' . json_last_error_msg();
            } elseif (!is_array($decoded) || empty($decoded)) {
                $errors['initial_state'] = 'Initial state must be a non-empty array';
            }
        }
        
        // Check if level number already exists (excluding current level)
        if (empty($errors)) {
            $existingLevel = fetch("SELECT id FROM levels WHERE level_number = ? AND id != ?", [$level['level_number'], $levelId]);
            if ($existingLevel) {
                $errors['level_number'] = 'Level number already exists';
            }
        }
        
        // If no errors, update the level
        if (empty($errors)) {
            try {
                $db = Database::getInstance();
                $db->beginTransaction();
                
                // Store old values for logging
                $oldValues = $level;
                
                // Update level
                $levelData = [
                    'name' => $name,
                    'description' => $description,
                    'difficulty' => $difficulty,
                    'bottles_count' => $bottlesCount,
                    'colors_count' => $colorsCount,
                    'bottle_capacity' => $bottleCapacity,
                    'initial_state' => $initialState,
                    'solution_moves' => $solutionMoves,
                    'time_limit' => $timeLimit,
                    'reward_coins' => $rewardCoins,
                    'is_active' => $isActive
                ];
                
                update('levels', $levelData, 'id = ?', [$levelId]);
                
                // Log activity
                logActivity('edit_level', 'levels', $levelId, $oldValues, $levelData);
                
                $db->commit();
                
                $success = 'Level updated successfully! <a href="' . ADMIN_URL . '/modules/levels.php">View all levels</a>';
                
                // Refresh level data
                $level = fetch("SELECT * FROM levels WHERE id = ?", [$levelId]);
                
            } catch (Exception $e) {
                $db->rollBack();
                $errors[] = 'Error updating level: ' . $e->getMessage();
            }
        }
    }
}
?>

<!-- Edit Level Content -->
<div class="edit-level-content">
    <!-- Display Notifications -->
    <?php displayNotifications(); ?>
    
    <!-- Page Header -->
    <div class="page-header">
        <h1 class="page-title">Edit Level</h1>
        <p class="page-description">Modify level #<?php echo $level['level_number']; ?> - <?php echo htmlspecialchars($level['name']); ?></p>
    </div>
    
    <!-- Success Message -->
    <?php if (!empty($success)): ?>
    <div class="alert alert-success">
        <span class="alert-icon">✅</span>
        <span class="alert-text"><?php echo $success; ?></span>
    </div>
    <?php endif; ?>
    
    <!-- Edit Level Form -->
    <div class="form-container">
        <form method="POST" class="level-form" data-validate>
            <!-- CSRF Token -->
            <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
            
            <!-- Level Info Display -->
            <div class="level-info-display">
                <div class="info-item">
                    <span class="info-label">Level Number:</span>
                    <span class="info-value">#<?php echo $level['level_number']; ?></span>
                </div>
                <div class="info-item">
                    <span class="info-label">Created:</span>
                    <span class="info-value"><?php echo formatDate($level['created_at']); ?></span>
                </div>
                <div class="info-item">
                    <span class="info-label">Times Played:</span>
                    <span class="info-value"><?php echo number_format($level['play_count'] ?? 0); ?></span>
                </div>
            </div>
            
            <!-- Basic Information -->
            <div class="form-section">
                <h3 class="section-title">Basic Information</h3>
                
                <div class="form-group">
                    <label for="name" class="form-label">Level Name *</label>
                    <input type="text" 
                           id="name" 
                           name="name" 
                           class="form-input" 
                           value="<?php echo htmlspecialchars($_POST['name'] ?? $level['name']); ?>"
                           placeholder="Enter level name"
                           required>
                    <?php if (isset($errors['name'])): ?>
                    <div class="field-error"><?php echo $errors['name']; ?></div>
                    <?php endif; ?>
                </div>
                
                <div class="form-group">
                    <label for="description" class="form-label">Description</label>
                    <textarea id="description" 
                              name="description" 
                              class="form-textarea" 
                              placeholder="Enter level description"
                              rows="3"><?php echo htmlspecialchars($_POST['description'] ?? $level['description'] ?? ''); ?></textarea>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="difficulty" class="form-label">Difficulty *</label>
                        <select id="difficulty" name="difficulty" class="form-select" required>
                            <option value="Easy" <?php echo (($_POST['difficulty'] ?? $level['difficulty']) === 'Easy') ? 'selected' : ''; ?>>Easy</option>
                            <option value="Medium" <?php echo (($_POST['difficulty'] ?? '') === 'Medium') ? 'selected' : ''; ?>>Medium</option>
                            <option value="Hard" <?php echo (($_POST['difficulty'] ?? '') === 'Hard') ? 'selected' : ''; ?>>Hard</option>
                            <option value="Expert" <?php echo (($_POST['difficulty'] ?? '') === 'Expert') ? 'selected' : ''; ?>>Expert</option>
                        </select>
                        <?php if (isset($errors['difficulty'])): ?>
                        <div class="field-error"><?php echo $errors['difficulty']; ?></div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="form-group">
                        <label for="reward_coins" class="form-label">Reward Coins</label>
                        <input type="number" 
                               id="reward_coins" 
                               name="reward_coins" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['reward_coins'] ?? $level['reward_coins']); ?>"
                               min="0"
                               max="999">
                    </div>
                </div>
            </div>
            
            <!-- Level Configuration -->
            <div class="form-section">
                <h3 class="section-title">Level Configuration</h3>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="bottles_count" class="form-label">Number of Bottles *</label>
                        <input type="number" 
                               id="bottles_count" 
                               name="bottles_count" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['bottles_count'] ?? $level['bottles_count']); ?>"
                               min="3"
                               max="20"
                               required>
                        <?php if (isset($errors['bottles_count'])): ?>
                        <div class="field-error"><?php echo $errors['bottles_count']; ?></div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="form-group">
                        <label for="colors_count" class="form-label">Number of Colors *</label>
                        <input type="number" 
                               id="colors_count" 
                               name="colors_count" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['colors_count'] ?? $level['colors_count']); ?>"
                               min="2"
                               max="10"
                               required>
                        <?php if (isset($errors['colors_count'])): ?>
                        <div class="field-error"><?php echo $errors['colors_count']; ?></div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="form-group">
                        <label for="bottle_capacity" class="form-label">Bottle Capacity *</label>
                        <input type="number" 
                               id="bottle_capacity" 
                               name="bottle_capacity" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['bottle_capacity'] ?? $level['bottle_capacity']); ?>"
                               min="3"
                               max="10"
                               required>
                        <?php if (isset($errors['bottle_capacity'])): ?>
                        <div class="field-error"><?php echo $errors['bottle_capacity']; ?></div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="solution_moves" class="form-label">Solution Moves (Optional)</label>
                        <input type="number" 
                               id="solution_moves" 
                               name="solution_moves" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['solution_moves'] ?? $level['solution_moves']); ?>"
                               min="0"
                               placeholder="Minimum moves to solve">
                    </div>
                    
                    <div class="form-group">
                        <label for="time_limit" class="form-label">Time Limit (Optional)</label>
                        <input type="number" 
                               id="time_limit" 
                               name="time_limit" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['time_limit'] ?? $level['time_limit']); ?>"
                               min="0"
                               placeholder="Seconds (0 = no limit)">
                    </div>
                </div>
            </div>
            
            <!-- Initial State -->
            <div class="form-section">
                <h3 class="section-title">Initial State (JSON) *</h3>
                <p class="section-description">Define the initial state of bottles as a JSON array. Each bottle is an array of colors.</p>
                
                <div class="form-group">
                    <label for="initial_state" class="form-label">Initial State JSON *</label>
                    <textarea id="initial_state" 
                              name="initial_state" 
                              class="form-textarea code-textarea" 
                              placeholder='[["red","red","blue","blue"], ["blue","red","blue","red"], [], []]'
                              rows="8"
                              required><?php echo htmlspecialchars($_POST['initial_state'] ?? $level['initial_state']); ?></textarea>
                    <?php if (isset($errors['initial_state'])): ?>
                    <div class="field-error"><?php echo $errors['initial_state']; ?></div>
                    <?php endif; ?>
                </div>
                
                <div class="json-tools">
                    <button type="button" class="btn btn-secondary" onclick="validateJSON()">
                        <span class="btn-icon">✅</span>
                        Validate JSON
                    </button>
                    <button type="button" class="btn btn-secondary" onclick="formatJSON()">
                        <span class="btn-icon">🎨</span>
                        Format JSON
                    </button>
                    <button type="button" class="btn btn-secondary" onclick="previewLevel()">
                        <span class="btn-icon">👁️</span>
                        Preview Level
                    </button>
                </div>
            </div>
            
            <!-- Status -->
            <div class="form-section">
                <h3 class="section-title">Status</h3>
                
                <div class="form-group">
                    <label class="checkbox-label">
                        <input type="checkbox" name="is_active" class="checkbox-input" <?php echo isset($_POST['is_active']) ? 'checked' : ($level['is_active'] ? 'checked' : ''); ?>>
                        <span class="checkbox-text">Level is active (available for players)</span>
                    </label>
                </div>
            </div>
            
            <!-- Form Actions -->
            <div class="form-actions">
                <button type="submit" class="btn btn-primary">
                    <span class="btn-icon">💾</span>
                    Update Level
                </button>
                <a href="<?php echo ADMIN_URL; ?>/modules/levels.php" class="btn btn-secondary">
                    <span class="btn-icon">✖</span>
                    Cancel
                </a>
                <button type="button" class="btn btn-danger" onclick="confirmDelete()">
                    <span class="btn-icon">🗑️</span>
                    Delete Level
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div id="deleteModal" class="modal" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Confirm Delete Level</h3>
            <button class="modal-close" onclick="closeModal()">&times;</button>
        </div>
        <div class="modal-body">
            <p>Are you sure you want to delete level <strong><?php echo htmlspecialchars($level['name']); ?></strong>?</p>
            <p class="warning-text">This action cannot be undone and will delete all game history for this level.</p>
        </div>
        <div class="modal-footer">
            <button class="btn btn-secondary" onclick="closeModal()">Cancel</button>
            <a href="<?php echo ADMIN_URL; ?>/modules/levels.php?delete=<?php echo $level['id']; ?>&csrf_token=<?php echo generateCSRFToken(); ?>" 
               class="btn btn-danger">Delete Level</a>
        </div>
    </div>
</div>

<!-- Level Preview Modal -->
<div id="previewModal" class="modal" style="display: none;">
    <div class="modal-content modal-large">
        <div class="modal-header">
            <h3>Level Preview</h3>
            <button class="modal-close" onclick="closePreviewModal()">&times;</button>
        </div>
        <div class="modal-body">
            <div id="previewContent">
                <!-- Preview content will be loaded here -->
            </div>
        </div>
        <div class="modal-footer">
            <button class="btn btn-secondary" onclick="closePreviewModal()">Close</button>
        </div>
    </div>
</div>

<script>
function validateJSON() {
    const textarea = document.getElementById('initial_state');
    const value = textarea.value;
    
    try {
        const parsed = JSON.parse(value);
        if (!Array.isArray(parsed)) {
            throw new Error('JSON must be an array');
        }
        showNotification('JSON is valid! Found ' + parsed.length + ' bottles.', 'success');
    } catch (e) {
        showNotification('Invalid JSON: ' + e.message, 'error');
    }
}

function formatJSON() {
    const textarea = document.getElementById('initial_state');
    const value = textarea.value;
    
    try {
        const parsed = JSON.parse(value);
        textarea.value = JSON.stringify(parsed, null, 2);
        showNotification('JSON formatted successfully!', 'success');
    } catch (e) {
        showNotification('Cannot format invalid JSON', 'error');
    }
}

function previewLevel() {
    const initialState = document.getElementById('initial_state').value;
    const bottlesCount = document.getElementById('bottles_count').value;
    const colorsCount = document.getElementById('colors_count').value;
    
    try {
        const parsed = JSON.parse(initialState);
        let previewHTML = '<div class="level-preview">';
        previewHTML += '<h4>Level Configuration</h4>';
        previewHTML += '<div class="preview-stats">';
        previewHTML += '<span>Bottles: ' + bottlesCount + '</span>';
        previewHTML += '<span>Colors: ' + colorsCount + '</span>';
        previewHTML += '</div>';
        previewHTML += '<h4>Initial State</h4>';
        previewHTML += '<div class="bottle-preview">';
        
        parsed.forEach((bottle, index) => {
            previewHTML += '<div class="bottle">';
            previewHTML += '<div class="bottle-label">Bottle ' + (index + 1) + '</div>';
            if (bottle.length === 0) {
                previewHTML += '<div class="bottle-empty">Empty</div>';
            } else {
                bottle.forEach(color => {
                    previewHTML += '<div class="bottle-color color-' + color + '">' + color + '</div>';
                });
            }
            previewHTML += '</div>';
        });
        
        previewHTML += '</div></div>';
        document.getElementById('previewContent').innerHTML = previewHTML;
        document.getElementById('previewModal').style.display = 'block';
    } catch (e) {
        showNotification('Cannot preview invalid JSON', 'error');
    }
}

function confirmDelete() {
    document.getElementById('deleteModal').style.display = 'block';
}

function closeModal() {
    document.getElementById('deleteModal').style.display = 'none';
}

function closePreviewModal() {
    document.getElementById('previewModal').style.display = 'none';
}

// Close modals when clicking outside
window.onclick = function(event) {
    const deleteModal = document.getElementById('deleteModal');
    const previewModal = document.getElementById('previewModal');
    
    if (event.target == deleteModal) {
        closeModal();
    }
    if (event.target == previewModal) {
        closePreviewModal();
    }
}
</script>

<style>
.level-preview {
    padding: 20px;
}

.preview-stats {
    display: flex;
    gap: 20px;
    margin-bottom: 20px;
}

.preview-stats span {
    background: #f0f0f0;
    padding: 5px 10px;
    border-radius: 4px;
}

.bottle-preview {
    display: flex;
    gap: 15px;
    flex-wrap: wrap;
}

.bottle {
    border: 2px solid #ddd;
    border-radius: 8px;
    padding: 10px;
    min-width: 100px;
    text-align: center;
}

.bottle-label {
    font-weight: bold;
    margin-bottom: 10px;
}

.bottle-empty {
    color: #999;
    font-style: italic;
    padding: 20px 0;
}

.bottle-color {
    padding: 5px;
    margin: 2px 0;
    border-radius: 3px;
    font-size: 12px;
    color: white;
}

.color-red { background: #ff4444; }
.color-blue { background: #4444ff; }
.color-green { background: #44ff44; }
.color-yellow { background: #ffff44; color: #333; }
.color-purple { background: #ff44ff; }
.color-orange { background: #ff8844; }
</style>

<?php
// Include footer
include_once ADMIN_PATH . '/includes/footer.php';
?>
