<?php
// Define admin access for security
define('ADMIN_ACCESS', true);

// Set page title
$page_title = 'Levels Management';

// Set current page for navigation
$current_page = 'levels.php';

// Include required files
require_once __DIR__ . '/../config/constants.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/functions.php';
require_once __DIR__ . '/../includes/security.php';

// Include header
include_once ADMIN_PATH . '/includes/header.php';

// Handle search and pagination
$search = sanitizeInput($_GET['search'] ?? '');
$difficulty = sanitizeInput($_GET['difficulty'] ?? '');
$page = max(1, intval($_GET['page'] ?? 1));
$limit = ITEMS_PER_PAGE;
$offset = ($page - 1) * $limit;

// Build query
$where = [];
$params = [];

if (!empty($search)) {
    $where[] = "(name LIKE ? OR description LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if (!empty($difficulty)) {
    $where[] = "difficulty = ?";
    $params[] = $difficulty;
}

$whereClause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';

// Get total levels count
$countQuery = "SELECT COUNT(*) as total FROM levels $whereClause";
$totalLevels = fetch($countQuery, $params)['total'];
$totalPages = ceil($totalLevels / $limit);

// Get levels with pagination
$query = "SELECT l.*, 
                 (SELECT COUNT(*) FROM game_plays WHERE level_id = l.id) as play_count,
                 (SELECT COUNT(*) FROM game_plays WHERE level_id = l.id AND status = 'won') as completion_count
          FROM levels l 
          $whereClause 
          ORDER BY l.level_number ASC 
          LIMIT ? OFFSET ?";

$levelsParams = array_merge($params, [$limit, $offset]);
$levels = fetchAll($query, $levelsParams);

// Handle level deletion
if (isset($_GET['delete']) && is_numeric($_GET['delete'])) {
    $levelId = intval($_GET['delete']);
    
    // Verify CSRF token
    if (!isset($_GET['csrf_token']) || !verifyCSRFToken($_GET['csrf_token'])) {
        showNotification('Invalid request token', 'error');
    } else {
        try {
            // Start transaction
            $db = Database::getInstance();
            $db->beginTransaction();
            
            // Delete level's game plays
            delete('game_plays', 'level_id = ?', [$levelId]);
            
            // Delete level
            delete('levels', 'id = ?', [$levelId]);
            
            // Log activity
            logActivity('delete_level', 'levels', $levelId, null, 'Level deleted by admin');
            
            $db->commit();
            showNotification('Level deleted successfully', 'success');
            
            // Redirect to refresh page
            header('Location: ' . ADMIN_URL . '/modules/levels.php');
            exit;
            
        } catch (Exception $e) {
            $db->rollBack();
            showNotification('Error deleting level: ' . $e->getMessage(), 'error');
        }
    }
}

// Get difficulty options for filter
$difficulties = ['Easy', 'Medium', 'Hard', 'Expert'];
?>

<!-- Levels Management Content -->
<div class="levels-content">
    <!-- Display Notifications -->
    <?php displayNotifications(); ?>
    
    <!-- Page Header -->
    <div class="page-header">
        <h1 class="page-title">Levels Management</h1>
        <p class="page-description">Manage all game levels and their statistics</p>
    </div>
    
    <!-- Search and Filters -->
    <div class="search-section">
        <form method="GET" class="search-form">
            <div class="search-filters">
                <div class="search-input-group">
                    <input type="text" 
                           name="search" 
                           class="search-input" 
                           placeholder="Search levels..."
                           value="<?php echo htmlspecialchars($search); ?>">
                    <select name="difficulty" class="filter-select">
                        <option value="">All Difficulties</option>
                        <?php foreach ($difficulties as $diff): ?>
                        <option value="<?php echo $diff; ?>" <?php echo $difficulty === $diff ? 'selected' : ''; ?>>
                            <?php echo $diff; ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                    <button type="submit" class="btn btn-primary">
                        <span class="btn-icon">🔍</span>
                        Search
                    </button>
                    <?php if (!empty($search) || !empty($difficulty)): ?>
                    <a href="<?php echo ADMIN_URL; ?>/modules/levels.php" class="btn btn-secondary">
                        <span class="btn-icon">✖</span>
                        Clear
                    </a>
                    <?php endif; ?>
                </div>
            </div>
        </form>
    </div>
    
    <!-- Quick Actions -->
    <div class="quick-actions">
        <a href="<?php echo ADMIN_URL; ?>/modules/add-level.php" class="btn btn-primary">
            <span class="btn-icon">➕</span>
            Add New Level
        </a>
        <button class="btn btn-secondary" onclick="exportLevels()">
            <span class="btn-icon">📥</span>
            Export Levels
        </button>
    </div>
    
    <!-- Levels Table -->
    <div class="table-container">
        <div class="table-header">
            <h2 class="table-title">All Levels (<?php echo number_format($totalLevels); ?>)</h2>
            <div class="table-stats">
                <span class="stat-item">
                    <span class="stat-label">Total Plays:</span>
                    <span class="stat-value"><?php echo number_format(array_sum(array_column($levels, 'play_count'))); ?></span>
                </span>
            </div>
        </div>
        
        <?php if (empty($levels)): ?>
        <div class="empty-state">
            <div class="empty-icon">🎮</div>
            <h3>No levels found</h3>
            <p><?php echo !empty($search) || !empty($difficulty) ? 'No levels match your search criteria.' : 'No levels have been created yet.'; ?></p>
            <a href="<?php echo ADMIN_URL; ?>/modules/add-level.php" class="btn btn-primary">
                <span class="btn-icon">➕</span>
                Create First Level
            </a>
        </div>
        <?php else: ?>
        <div class="data-table">
            <table class="levels-table">
                <thead>
                    <tr>
                        <th>Level</th>
                        <th>Name</th>
                        <th>Difficulty</th>
                        <th>Configuration</th>
                        <th>Statistics</th>
                        <th>Completion Rate</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($levels as $level): ?>
                    <tr>
                        <td>
                            <div class="level-number">
                                <span class="level-badge">#<?php echo $level['level_number']; ?></span>
                            </div>
                        </td>
                        <td>
                            <div class="level-info">
                                <div class="level-name"><?php echo htmlspecialchars($level['name']); ?></div>
                                <div class="level-description"><?php echo htmlspecialchars($level['description'] ?? ''); ?></div>
                            </div>
                        </td>
                        <td>
                            <span class="difficulty-badge difficulty-<?php echo strtolower($level['difficulty']); ?>">
                                <?php echo $level['difficulty']; ?>
                            </span>
                        </td>
                        <td>
                            <div class="level-config">
                                <div class="config-item">
                                    <span class="config-label">Bottles:</span>
                                    <span class="config-value"><?php echo $level['bottles_count']; ?></span>
                                </div>
                                <div class="config-item">
                                    <span class="config-label">Colors:</span>
                                    <span class="config-value"><?php echo $level['colors_count']; ?></span>
                                </div>
                                <div class="config-item">
                                    <span class="config-label">Moves:</span>
                                    <span class="config-value"><?php echo $level['solution_moves']; ?></span>
                                </div>
                            </div>
                        </td>
                        <td>
                            <div class="level-stats">
                                <div class="stat-row">
                                    <span class="stat-label">Plays:</span>
                                    <span class="stat-value"><?php echo number_format($level['play_count']); ?></span>
                                </div>
                                <div class="stat-row">
                                    <span class="stat-label">Wins:</span>
                                    <span class="stat-value"><?php echo number_format($level['completion_count']); ?></span>
                                </div>
                                <div class="stat-row">
                                    <span class="stat-label">Reward:</span>
                                    <span class="stat-value"><?php echo $level['reward_coins']; ?> coins</span>
                                </div>
                            </div>
                        </td>
                        <td>
                            <?php
                            $completionRate = $level['play_count'] > 0 ? round(($level['completion_count'] / $level['play_count']) * 100, 1) : 0;
                            $rateClass = $completionRate >= 70 ? 'high' : ($completionRate >= 40 ? 'medium' : 'low');
                            ?>
                            <div class="completion-rate">
                                <div class="rate-bar">
                                    <div class="rate-fill rate-<?php echo $rateClass; ?>" style="width: <?php echo $completionRate; ?>%"></div>
                                </div>
                                <span class="rate-text"><?php echo $completionRate; ?>%</span>
                            </div>
                        </td>
                        <td>
                            <span class="status-badge <?php echo $level['is_active'] ? 'active' : 'inactive'; ?>">
                                <?php echo $level['is_active'] ? 'Active' : 'Inactive'; ?>
                            </span>
                        </td>
                        <td>
                            <div class="action-buttons">
                                <button class="btn btn-sm btn-info" 
                                        onclick="previewLevel(<?php echo $level['id']; ?>)"
                                        title="Preview Level">
                                    <span class="btn-icon">👁️</span>
                                </button>
                                <a href="<?php echo ADMIN_URL; ?>/modules/edit-level.php?id=<?php echo $level['id']; ?>" 
                                   class="btn btn-sm btn-warning" title="Edit Level">
                                    <span class="btn-icon">✏️</span>
                                </a>
                                <button class="btn btn-sm btn-<?php echo $level['is_active'] ? 'secondary' : 'success'; ?>" 
                                        onclick="toggleLevelStatus(<?php echo $level['id']; ?>, <?php echo $level['is_active'] ? '0' : '1'; ?>)"
                                        title="<?php echo $level['is_active'] ? 'Deactivate' : 'Activate'; ?>">
                                    <span class="btn-icon"><?php echo $level['is_active'] ? '⏸️' : '▶️'; ?></span>
                                </button>
                                <button class="btn btn-sm btn-danger" 
                                        onclick="confirmDelete(<?php echo $level['id']; ?>, '<?php echo htmlspecialchars($level['name']); ?>')"
                                        title="Delete Level">
                                    <span class="btn-icon">🗑️</span>
                                </button>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Pagination -->
        <?php if ($totalPages > 1): ?>
        <div class="pagination">
            <?php
            $currentUrl = ADMIN_URL . '/modules/levels.php';
            $queryParams = [];
            if (!empty($search)) $queryParams['search'] = $search;
            if (!empty($difficulty)) $queryParams['difficulty'] = $difficulty;
            
            $urlString = $currentUrl . (!empty($queryParams) ? '?' . http_build_query($queryParams) : '');
            ?>
            
            <?php if ($page > 1): ?>
            <a href="<?php echo $urlString . (strpos($urlString, '?') !== false ? '&' : '?') . 'page=' . ($page - 1); ?>" 
               class="btn btn-pagination">« Previous</a>
            <?php endif; ?>
            
            <?php
            $startPage = max(1, $page - 2);
            $endPage = min($totalPages, $page + 2);
            
            for ($i = $startPage; $i <= $endPage; $i++):
            ?>
            <a href="<?php echo $urlString . (strpos($urlString, '?') !== false ? '&' : '?') . 'page=' . $i; ?>" 
               class="btn btn-pagination <?php echo $i == $page ? 'active' : ''; ?>">
                <?php echo $i; ?>
            </a>
            <?php endfor; ?>
            
            <?php if ($page < $totalPages): ?>
            <a href="<?php echo $urlString . (strpos($urlString, '?') !== false ? '&' : '?') . 'page=' . ($page + 1); ?>" 
               class="btn btn-pagination">Next »</a>
            <?php endif; ?>
        </div>
        <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div id="deleteModal" class="modal" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Confirm Delete Level</h3>
            <button class="modal-close" onclick="closeModal()">&times;</button>
        </div>
        <div class="modal-body">
            <p>Are you sure you want to delete level <strong id="deleteLevelName"></strong>?</p>
            <p class="warning-text">This action cannot be undone and will delete all game history for this level.</p>
        </div>
        <div class="modal-footer">
            <button class="btn btn-secondary" onclick="closeModal()">Cancel</button>
            <a id="deleteConfirmBtn" href="#" class="btn btn-danger">Delete Level</a>
        </div>
    </div>
</div>

<!-- Level Preview Modal -->
<div id="previewModal" class="modal" style="display: none;">
    <div class="modal-content modal-large">
        <div class="modal-header">
            <h3>Level Preview</h3>
            <button class="modal-close" onclick="closePreviewModal()">&times;</button>
        </div>
        <div class="modal-body">
            <div id="previewContent">
                <!-- Preview content will be loaded here -->
            </div>
        </div>
        <div class="modal-footer">
            <button class="btn btn-secondary" onclick="closePreviewModal()">Close</button>
        </div>
    </div>
</div>

<script>
function confirmDelete(levelId, levelName) {
    document.getElementById('deleteLevelName').textContent = levelName;
    document.getElementById('deleteConfirmBtn').href = 
        '<?php echo ADMIN_URL; ?>/modules/levels.php?delete=' + levelId + '&csrf_token=<?php echo generateCSRFToken(); ?>';
    document.getElementById('deleteModal').style.display = 'block';
}

function closeModal() {
    document.getElementById('deleteModal').style.display = 'none';
}

function previewLevel(levelId) {
    fetch('<?php echo ADMIN_URL; ?>/modules/preview-level.php?id=' + levelId)
        .then(response => response.text())
        .then(html => {
            document.getElementById('previewContent').innerHTML = html;
            document.getElementById('previewModal').style.display = 'block';
        })
        .catch(error => {
            console.error('Error loading preview:', error);
            showNotification('Error loading level preview', 'error');
        });
}

function closePreviewModal() {
    document.getElementById('previewModal').style.display = 'none';
}

function toggleLevelStatus(levelId, newStatus) {
    const url = '<?php echo ADMIN_URL; ?>/modules/toggle-level.php';
    const formData = new FormData();
    formData.append('level_id', levelId);
    formData.append('is_active', newStatus);
    formData.append('csrf_token', '<?php echo generateCSRFToken(); ?>');
    
    fetch(url, {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('Level status updated successfully', 'success');
            location.reload();
        } else {
            showNotification(data.message || 'Error updating level status', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('Error updating level status', 'error');
    });
}

function exportLevels() {
    window.location.href = '<?php echo ADMIN_URL; ?>/modules/export-levels.php';
}

// Close modals when clicking outside
window.onclick = function(event) {
    const deleteModal = document.getElementById('deleteModal');
    const previewModal = document.getElementById('previewModal');
    
    if (event.target == deleteModal) {
        closeModal();
    }
    if (event.target == previewModal) {
        closePreviewModal();
    }
}
</script>

<?php
// Include footer
include_once ADMIN_PATH . '/includes/footer.php';
?>
