<?php
/**
 * Quick Setup Script for Water Sort Game Admin Panel
 * This script will help you set up the database and admin user
 */

// Define admin access for security
define('ADMIN_ACCESS', true);

// Include required files
require_once __DIR__ . '/config/constants.php';
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/config/functions.php';

// Check if setup is already complete
if (isSetupComplete()) {
    echo "<h1>Setup Already Complete</h1>";
    echo "<p>The admin panel is already set up. You can <a href='auth/login.php'>login here</a>.</p>";
    echo "<p>Default credentials: admin / admin123</p>";
    exit;
}

// Handle setup form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $db_host = $_POST['db_host'] ?? 'localhost';
    $db_name = $_POST['db_name'] ?? 'water_sort_admin';
    $db_user = $_POST['db_user'] ?? 'root';
    $db_pass = $_POST['db_pass'] ?? '';
    $admin_username = $_POST['admin_username'] ?? 'admin';
    $admin_password = $_POST['admin_password'] ?? 'admin123';
    $admin_email = $_POST['admin_email'] ?? 'admin@example.com';
    
    // Test database connection
    try {
        $pdo = new PDO("mysql:host=$db_host", $db_user, $db_pass);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        // Create database
        $pdo->exec("CREATE DATABASE IF NOT EXISTS `$db_name` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        $pdo->exec("USE `$db_name`");
        
        // Read and execute SQL setup file
        $sql = file_get_contents(__DIR__ . '/database/setup.sql');
        $sql = str_replace('`water_sort_admin`', "`$db_name`", $sql);
        
        // Enable buffered queries to prevent the error
        $pdo->setAttribute(PDO::MYSQL_ATTR_USE_BUFFERED_QUERY, true);
        
        // Execute SQL statements one by one
        $statements = array_filter(array_map('trim', preg_split('/;\s*\n/', $sql)));
        
        foreach ($statements as $statement) {
            if (!empty($statement) && !preg_match('/^(--|#|\/\*)/', $statement)) {
                try {
                    $pdo->exec($statement);
                } catch (Exception $e) {
                    // Continue even if some statements fail (like CREATE DATABASE if it exists)
                    if (strpos($e->getMessage(), 'already exists') === false) {
                        throw $e;
                    }
                }
            }
        }
        
        // Insert admin user
        $hashed_password = password_hash($admin_password, PASSWORD_ARGON2ID);
        $stmt = $pdo->prepare("INSERT INTO admins (username, email, password, role) VALUES (?, ?, ?, 'admin')");
        $stmt->execute([$admin_username, $admin_email, $hashed_password]);
        
        // Update config file with new database settings
        updateConfigFile($db_host, $db_name, $db_user, $db_pass);
        
        echo "<h1>Setup Complete!</h1>";
        echo "<p>✅ Database created successfully</p>";
        echo "<p>✅ Admin user created successfully</p>";
        echo "<p>✅ Configuration updated</p>";
        echo "<p>You can now <a href='auth/login.php'>login to the admin panel</a></p>";
        echo "<p>Login credentials:<br>";
        echo "Username: <strong>$admin_username</strong><br>";
        echo "Password: <strong>$admin_password</strong></p>";
        
        // Create setup completion flag
        file_put_contents(__DIR__ . '/config/setup_complete', date('Y-m-d H:i:s'));
        
    } catch (Exception $e) {
        echo "<h1>Setup Error</h1>";
        echo "<p>❌ Error: " . htmlspecialchars($e->getMessage()) . "</p>";
        echo "<p>Please check your database credentials and try again.</p>";
        echo "<p><a href='setup.php'>Try Again</a></p>";
    }
    exit;
}

/**
 * Check if setup is already complete
 */
function isSetupComplete() {
    return file_exists(__DIR__ . '/config/setup_complete');
}

/**
 * Update config file with database settings
 */
function updateConfigFile($host, $name, $user, $pass) {
    $configFile = __DIR__ . '/config/constants.php';
    $content = file_get_contents($configFile);
    
    // Update database constants
    $content = preg_replace("/define\('DB_HOST', '[^']+'\)/", "define('DB_HOST', '$host')", $content);
    $content = preg_replace("/define\('DB_NAME', '[^']+'\)/", "define('DB_NAME', '$name')", $content);
    $content = preg_replace("/define\('DB_USER', '[^']+'\)/", "define('DB_USER', '$user')", $content);
    $content = preg_replace("/define\('DB_PASS', '[^']+'\)/", "define('DB_PASS', '$pass')", $content);
    
    file_put_contents($configFile, $content);
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Setup - Water Sort Game Admin Panel</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; background: #f5f5f5; }
        .container { max-width: 600px; margin: 50px auto; background: white; border-radius: 10px; box-shadow: 0 10px 30px rgba(0,0,0,0.1); overflow: hidden; }
        .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 30px; text-align: center; }
        .header h1 { font-size: 28px; margin-bottom: 10px; }
        .header p { opacity: 0.9; }
        .content { padding: 40px; }
        .form-group { margin-bottom: 25px; }
        .form-group label { display: block; margin-bottom: 8px; font-weight: 600; color: #333; }
        .form-group input { width: 100%; padding: 12px; border: 2px solid #e1e5e9; border-radius: 6px; font-size: 14px; transition: border-color 0.3s; }
        .form-group input:focus { outline: none; border-color: #667eea; }
        .form-section { margin-bottom: 30px; }
        .form-section h3 { color: #333; margin-bottom: 20px; padding-bottom: 10px; border-bottom: 2px solid #f0f0f0; }
        .btn { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 14px 30px; border: none; border-radius: 6px; font-size: 16px; font-weight: 600; cursor: pointer; transition: transform 0.2s; }
        .btn:hover { transform: translateY(-2px); }
        .info-box { background: #f8f9fa; border-left: 4px solid #667eea; padding: 15px; margin-bottom: 20px; border-radius: 4px; }
        .info-box p { margin: 0; color: #666; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>💧 Water Sort Game</h1>
            <p>Admin Panel Setup</p>
        </div>
        
        <div class="content">
            <div class="info-box">
                <p>This setup wizard will help you configure the database and create your admin account.</p>
            </div>
            
            <form method="POST" action="">
                <div class="form-section">
                    <h3>📊 Database Configuration</h3>
                    
                    <div class="form-group">
                        <label for="db_host">Database Host</label>
                        <input type="text" id="db_host" name="db_host" value="localhost" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="db_name">Database Name</label>
                        <input type="text" id="db_name" name="db_name" value="water_sort_admin" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="db_user">Database Username</label>
                        <input type="text" id="db_user" name="db_user" value="root" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="db_pass">Database Password</label>
                        <input type="password" id="db_pass" name="db_pass" value="">
                    </div>
                </div>
                
                <div class="form-section">
                    <h3>👤 Admin Account</h3>
                    
                    <div class="form-group">
                        <label for="admin_username">Admin Username</label>
                        <input type="text" id="admin_username" name="admin_username" value="admin" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="admin_email">Admin Email</label>
                        <input type="email" id="admin_email" name="admin_email" value="admin@example.com" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="admin_password">Admin Password</label>
                        <input type="password" id="admin_password" name="admin_password" value="admin123" required>
                    </div>
                </div>
                
                <button type="submit" class="btn">🚀 Start Setup</button>
            </form>
        </div>
    </div>
</body>
</html>
